"""
    LICENSE
    =======

    Copyright (c) 2004, Randall Smith
    All rights reserved.

    Redistribution and use in source and binary forms, with or without 
    modification, are permitted provided that the following conditions are met:

        * Redistributions of source code must retain the above copyright 
          notice, this list of conditions and the following disclaimer.
        * Redistributions in binary form must reproduce the above copyright 
          notice, this list of conditions and the following disclaimer in the 
          documentation and/or other materials provided with the distribution.
        * Neither the name of the <ORGANIZATION> nor the names of its 
          contributors may be used to endorse or promote products derived from 
          this software without specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
    AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
    IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
    ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
    LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
    CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
    SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
    CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
    ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
    POSSIBILITY OF SUCH DAMAGE.

"""
try:
    import hashlib
    oldmd5 = False
except:
    import md5
    oldmd5 = True
# import from hashlib if possible, if older version use md5
import binascii
import struct
import socket
import string

# Protocol Type Functions

def Int32(pyint):
    # Make sure this is a 32 bit integer then:
    return struct.pack('!i', pyint)

def Int32U(struct_int):
    # Unpack integer from struct.
    # This seems to be a problem under 2.6
    if len(struct_int)<>4:
		print "Not 4"
		print len(struct_int)
		print struct_int
    i = struct.unpack('I', struct_int) # this was i
    return socket.ntohl(i[0])

def Int16(pyint):
    # Make sure this is a 16 bit integer then:
    return struct.pack('!h', pyint)

def Int16U(struct_int):
    # Unpack integer from struct.
    return socket.ntohs(struct.unpack('H', struct_int)[0])

def Int32Array(intlist):
    return struct.pack('!%ii' % len(intlist), *intlist)

def Int16Array(intlist):
    return struct.pack('!%ih' % len(intlist), *intlist)

# Messges.

class Message(object):
    """Postgresql Message.
    
    Message type is determined and attributes are created. 
    Must be ready to accept asynchronous messages which can be:
        NoticeResponse or
        ParameterStatus
            server_version
            client_encoding
            is_superuser
            session_authorization
            DateStyle
    """

    def __init__(self, server_response=None):
        self.format_name = self.__class__.__name__
        self.server_message = None
        self.server_response = server_response
        if self.server_response is not None:
            self.identity = server_response[0]
            self.length = Int32U(server_response[1:5])

    def prepareForServer(self):
        """Create message to send to server.
        
        Should be overridden.

        """
        raise NotImplementedError

    def prepareForClient(self):
        """Read response from server and create message.
        
        Should be overridden.

        """
        raise NotImplementedError

class AuthenticationOk(Message):
    """
    Definition from
    http://www.postgresql.org/docs/7.4/interactive/protocol-message-formats.html

    AuthenticationOk (B)
        Byte1('R')
            Identifies the message as an authentication request.
        Int32(8)
            Length of message contents in bytes, including self.
        Int32(0)
            Specifies that the authentication was successful.

    """

    def __init__(self, server_response):
        Message.__init__(self, server_response)

class AuthRequestMessage(Message):
    """Base class for all authentication requests.

    """

    def __init__(self, server_response):
        Message.__init__(self, server_response)
        self.specifier = None
        self.salt = None
        if self.server_response is not None:
            self.specifier = Int32U(self.server_response[5:9])

class AuthenticationKerberosV4(AuthRequestMessage):
    """
    Definition from
    http://www.postgresql.org/docs/7.4/interactive/protocol-message-formats.html

    AuthenticationKerberosV4 (B)
        Byte1('R')
            Identifies the message as an authentication request.
        Int32(8)
            Length of message contents in bytes, including self.
        Int32(1)
            Specifies that Kerberos V4 authentication is required.

    """

    def __init__(self, server_response):
        AuthRequestMessage.__init__(self, server_response)

class AuthenticationKerberosV5(AuthRequestMessage):
    """
    Definition from
    http://www.postgresql.org/docs/7.4/interactive/protocol-message-formats.html

    AuthenticationKerberosV5 (B)
        Byte1('R')
            Identifies the message as an authentication request.
        Int32(8)
            Length of message contents in bytes, including self.
        Int32(2)
            Specifies that Kerberos V5 authentication is required.

    """

    def __init__(self, server_response):
        AuthRequestMessage.__init__(self, server_response)

class AuthenticationCleartextPassword(AuthRequestMessage):
    """
    Definition from
    http://www.postgresql.org/docs/7.4/interactive/protocol-message-formats.html

    AuthenticationCleartextPassword (B)
        Byte1('R')
            Identifies the message as an authentication request.
        Int32(8)
            Length of message contents in bytes, including self.
        Int32(3)
            Specifies that a clear-text password is required.

    """

    def __init__(self, server_response):
        AuthRequestMessage.__init__(self, server_response)

class AuthenticationCryptPassword(AuthRequestMessage):
    """
    Definition from
    http://www.postgresql.org/docs/7.4/interactive/protocol-message-formats.html

    AuthenticationCryptPassword (B)
        Byte1('R')
            Identifies the message as an authentication request.
        Int32(10)
            Length of message contents in bytes, including self.
        Int32(4)
            Specifies that a crypt()-encrypted password is required.
        Byte2
            The salt to use when encrypting the password. 

    """

    def __init__(self, server_response):
        AuthRequestMessage.__init__(self, server_response)

class AuthenticationMD5Password(AuthRequestMessage):
    """
    Definition from
    http://www.postgresql.org/docs/7.4/interactive/protocol-message-formats.html

    Byte1('R')
        Identifies the message as an authentication request.
    Int32(12)
        Length of message contents in bytes, including self.
    Int32(5)
        Specifies that an MD5-encrypted password is required.
    Byte4
        The salt to use when encrypting the password.

    """

    def __init__(self, server_response):
        AuthRequestMessage.__init__(self, server_response)
        self.prepareForClient()

    def prepareForClient(self):
        server_response = self.server_response
        self.salt = server_response[9:13]

class AuthenticationSCMCredential(AuthRequestMessage):
    """
    Definition from
    http://www.postgresql.org/docs/7.4/interactive/protocol-message-formats.html

    AuthenticationSCMCredential (B)
        Byte1('R')
            Identifies the message as an authentication request.
        Int32(8)
            Length of message contents in bytes, including self.
        Int32(6)
            Specifies that an SCM credentials message is required.

    """

    def __init__(self, server_response):
        AuthRequestMessage.__init__(self, server_response)

class BackendKeyData(Message):
    """
    Definition from
    http://www.postgresql.org/docs/7.4/interactive/protocol-message-formats.html

    BackendKeyData (B)
        Byte1('K')
            Identifies the message as cancellation key data. The frontend must
            save these values if it wishes to be able to issue CancelRequest
            messages later.
        Int32(12)
            Length of message contents in bytes, including self.
        Int32
            The process ID of this backend.
        Int32
            The secret key of this backend.

    """

    def __init__(self, server_response):
        Message.__init__(self, server_response)
        self.process_id = None
        self.secret_key = None
        self.prepareForClient()

    def prepareForClient(self):
        server_response = self.server_response
        self.process_id = Int32U(server_response[5:9])
        self.secret_key = Int32U(server_response[9:13])

class BindComplete(Message):
    """
    Definition from
    http://www.postgresql.org/docs/7.4/interactive/protocol-message-formats.html

    BindComplete (B)
        Byte1('2')
            Identifies the message as a Bind-complete indicator.
        Int32(4)
            Length of message contents in bytes, including self.

    """

    def __init__(self, server_response):
        Message.__init__(self, server_response)

class CloseComplete(Message):
    """
    Definition from
    http://www.postgresql.org/docs/7.4/interactive/protocol-message-formats.html

    CloseComplete (B)
        Byte1('3')
            Identifies the message as a Close-complete indicator.
        Int32(4)
            Length of message contents in bytes, including self. 

    """

    def __init__(self, server_response):
        Message.__init__(self, server_response)

class CommandComplete(Message):
    """
    Definition from
    http://www.postgresql.org/docs/7.4/interactive/protocol-message-formats.html

    CommandComplete (B)

        Byte1('C')

            Identifies the message as a command-completed response.
        Int32

            Length of message contents in bytes, including self.
        String

            The command tag. This is usually a single word that identifies 
            which SQL command was completed.

            For an INSERT command, the tag is INSERT oid rows, where rows is 
            the number of rows inserted. oid is the object ID of the inserted 
            row if rows is 1 and the target table has OIDs; otherwise oid is 0.

            For a DELETE command, the tag is DELETE rows where rows is the 
            number of rows deleted.

            For an UPDATE command, the tag is UPDATE rows where rows is the 
            number of rows updated.

            For a MOVE command, the tag is MOVE rows where rows is the number 
            of rows the cursor's position has been changed by.

            For a FETCH command, the tag is FETCH rows where rows is the 
            number of rows that have been retrieved from the cursor.

    """

    def __init__(self, server_response):
        Message.__init__(self, server_response)

class CopyDone(Message):
    """
    Definition from
    http://www.postgresql.org/docs/7.4/interactive/protocol-message-formats.html

    CopyData (F & B)
        Byte1('d')

            Identifies the message as COPY data.
        Int32
            Length of message contents in bytes, including self.
        Byten
            Data that forms part of a COPY data stream. Messages sent from the 
            backend will always correspond to single data rows, but messages 
            sent by frontends may divide the data stream arbitrarily.

    """

    def __init__(self, server_response=None):
        Message.__init__(self, server_response)

class CopyFail(Message):
    """
    Definition from
    http://www.postgresql.org/docs/7.4/interactive/protocol-message-formats.html

    CopyFail (F)
        Byte1('f')
            Identifies the message as a COPY-failure indicator.
        Int32
            Length of message contents in bytes, including self.
        String
            An error message to report as the cause of failure.

    """

    def __init__(self):
        Message.__init__(self)

class CopyInResponse(Message):
    """
    Definition from
    http://www.postgresql.org/docs/7.4/interactive/protocol-message-formats.html

    CopyInResponse (B)
        Byte1('G')
            Identifies the message as a Start Copy In response. The frontend 
            must now send copy-in data (if not prepared to do so, send a 
            CopyFail message).
        Int32
            Length of message contents in bytes, including self.
        Int8
            0 indicates the overall copy format is textual (rows separated by 
            newlines, columns separated by separator characters, etc). 1 
            indicates the overall copy format is binary (similar to DataRow 
            format). See COPY for more information.
        Int16
            The number of columns in the data to be copied (denoted N below).
        Int16[N]
            The format codes to be used for each column. Each must presently 
            be zero (text) or one (binary). All must be zero if the overall 
            copy format is textual.

    """

    def __init__(self, server_response):
        Message.__init__(self, server_response)

class CopyOutResponse(Message):
    """
    Definition from
    http://www.postgresql.org/docs/7.4/interactive/protocol-message-formats.html

    CopyOutResponse (B)
        Byte1('H')
            Identifies the message as a Start Copy Out response. This message 
            will be followed by copy-out data.
        Int32
            Length of message contents in bytes, including self.
        Int8
            0 indicates the overall copy format is textual (rows separated by 
            newlines, columns separated by separator characters, etc). 1 
            indicates the overall copy format is binary (similar to DataRow 
            format). See COPY for more information.
        Int16
            The number of columns in the data to be copied (denoted N below).
        Int16[N]
            The format codes to be used for each column. Each must presently 
            be zero (text) or one (binary). All must be zero if the overall 
            copy format is textual.

    """

    def __init__(self, server_response):
        Message.__init__(self, server_response)

class DataRow(Message):
    """
    Definition from
    http://www.postgresql.org/docs/7.4/interactive/protocol-message-formats.html

    DataRow (B)
        Byte1('D')
            Identifies the message as a data row.
        Int32
            Length of message contents in bytes, including self.
        Int16
            The number of column values that follow (possibly zero).
        Next, the following pair of fields appear for each column:
        Int32
            The length of the column value, in bytes (this count does not 
            include itself). Can be zero. As a special case, -1 indicates a 
            NULL column value. No value bytes follow in the NULL case.
        Byten
            The value of the column, in the format indicated by the associated 
            format code. n is the above length.

    """

    def __init__(self, server_response):
        Message.__init__(self, server_response)
        self.colnum = None
        self.rowdata = None
        self.prepareForServer()

    def prepareForServer(self):
        server_response = self.server_response
        self.colnum = Int16U(server_response[5:7])
        rowdata = []
        self.rowdata = rowdata
        position = 7
        for col_id in range(self.colnum):
            if len(server_response[position:position + 4])==4:
                length = Int32U(server_response[position:position + 4])
                if length==4294967295:length=-1 # not a good idea !!, Int32 gives only good result when I instead of i
                # This gives false length sometimes, this fixxes this as long as there are no fields of this length
                # the change of this I don't know
            else:
                length = -1
            position = position + 4
            if length == -1:
                value = None
                length = 0
            else:
                value = server_response[position:position + length]
            rowdata.append(value)
            position = position + length

class EmptyQueryResponse(Message):
    """
    Definition from
    http://www.postgresql.org/docs/7.4/interactive/protocol-message-formats.html

    EmptyQueryResponse (B)
        Byte1('I')
            Identifies the message as a response to an empty query string. 
            (This substitutes for CommandComplete.)
        Int32(4)
            Length of message contents in bytes, including self.

    """

    def __init__(self, server_response):
        Message.__init__(self, server_response)

class ErrorResponse(Message):
    """
    Definition from
    http://www.postgresql.org/docs/7.4/interactive/protocol-message-formats.html

    ErrorResponse (B)
        Byte1('E')
            Identifies the message as an error.
        Int32
            Length of message contents in bytes, including self.
        The message body consists of one or more identified fields, followed 
        by a zero byte as a terminator. Fields may appear in any order. For 
        each field there is the following:
        Byte1
            A code identifying the field type; if zero, this is the message 
            terminator and no string follows. The presently defined field 
            types are listed in Section 44.5. Since more field types may be 
            added in future, frontends should silently ignore fields of 
            unrecognized type.
        String
            The field value.

    """

    def __init__(self, server_response):
        Message.__init__(self, server_response)
        self.prepareForClient()

    def prepareForClient(self):
        server_response = self.server_response
        parameters = server_response[20:]
        param_parts = parameters.split(chr(0))
        self.param_value = param_parts[0]

class FunctionCallResponse(Message):
    """
    Definition from
    http://www.postgresql.org/docs/7.4/interactive/protocol-message-formats.html

    FunctionCallResponse (B)
        Byte1('V')
            Identifies the message as a function call result.
        Int32
            Length of message contents in bytes, including self.
        Int32
            The length of the function result value, in bytes (this count does 
            not include itself). Can be zero. As a special case, -1 indicates 
            a NULL function result. No value bytes follow in the NULL case.
        Byten
            The value of the function result, in the format indicated by the 
            associated format code. n is the above length.

    """

    def __init__(self, server_response):
        Message.__init__(self, server_response)

class NoData(Message):
    """
    Definition from
    http://www.postgresql.org/docs/7.4/interactive/protocol-message-formats.html

    NoData (B)

        Byte1('n')
            Identifies the message as a no-data indicator.
        Int32(4)
            Length of message contents in bytes, including self.

    """

    def __init__(self, server_response):
        Message.__init__(self, server_response)

class NoticeResponse(Message):
    """
    Definition from
    http://www.postgresql.org/docs/7.4/interactive/protocol-message-formats.html

    NoticeResponse (B)
        Byte1('N')

            Identifies the message as a notice.
        Int32
            Length of message contents in bytes, including self.
        The message body consists of one or more identified fields, followed 
        by a zero byte as a terminator. Fields may appear in any order. For 
        each field there is the following:
        Byte1
            A code identifying the field type; if zero, this is the message 
            terminator and no string follows. The presently defined field 
            types are listed in Section 44.5. Since more field types may be 
            added in future, frontends should silently ignore fields of 
            unrecognized type.
        String
            The field value.NoticeResponse (B)
        Byte1('N')
            Identifies the message as a notice.
        Int32
            Length of message contents in bytes, including self.
        The message body consists of one or more identified fields, followed 
        by a zero byte as a terminator. Fields may appear in any order. For 
        each field there is the following:
        Byte1
            A code identifying the field type; if zero, this is the message 
            terminator and no string follows. The presently defined field 
            types are listed in Section 44.5. Since more field types may be 
            added in future, frontends should silently ignore fields of 
            unrecognized type.
        String
            The field value.

    """

    def __init__(self, server_response):
        Message.__init__(self, server_response)

class NotificationResponse(Message):
    """
    Definition from
    http://www.postgresql.org/docs/7.4/interactive/protocol-message-formats.html
    
    NotificationResponse (B)
        Byte1('A')
            Identifies the message as a notification response.
        Int32
            Length of message contents in bytes, including self.
        Int32
            The process ID of the notifying backend process.
        String
            The name of the condition that the notify has been raised on.
        String
            Additional information passed from the notifying process. 
            (Currently, this feature is unimplemented so the field is always 
            an empty string.)

    """

    def __init__(self, server_response):
        Message.__init__(self, server_response)

class ParameterDescription(Message):
    """
    Definition from
    http://www.postgresql.org/docs/7.4/interactive/protocol-message-formats.html
    
    ParameterDescription (B)
        Byte1('t')
            Identifies the message as a parameter description.
        Int32
            Length of message contents in bytes, including self.
        Int16
            The number of parameters used by the statement (may be zero).
        Then, for each parameter, there is the following:
        Int32
            Specifies the object ID of the parameter data type.

    """

    def __init__(self, server_response):
        Message.__init__(self, server_response)

class ParameterStatus(Message):
    """
    Definition from
    http://www.postgresql.org/docs/7.4/interactive/protocol-message-formats.html

    ParameterStatus (B)
        Byte1('S')
            Identifies the message as a run-time parameter status report.
        Int32
            Length of message contents in bytes, including self.
        String
            The name of the run-time parameter being reported.
        String
            The current value of the parameter.

    Async ParameterStatus Types:
    server_version
    client_encoding
    is_superuser
    session_authorization
    DateStyle

    """

    def __init__(self, server_response):
        Message.__init__(self, server_response)
        self.param_name = None
        self.param_value = None
        self.prepareForClient()

    def prepareForClient(self):
        server_response = self.server_response
        parameters = server_response[5:]
        param_parts = parameters.split(chr(0))
        self.param_name = param_parts[0]
        self.param_value = param_parts[1]

class ParseComplete(Message):
    """
    Definition from
    http://www.postgresql.org/docs/7.4/interactive/protocol-message-formats.html

    ParseComplete (B)
        Byte1('1')
            Identifies the message as a Parse-complete indicator.
        Int32(4)
            Length of message contents in bytes, including self.

    """

    def __init__(self, server_response):
        Message.__init__(self, server_response)

class PortalSuspended(Message):
    """
    Definition from
    http://www.postgresql.org/docs/7.4/interactive/protocol-message-formats.html

    ParseComplete (B)
        Byte1('1')
            Identifies the message as a Parse-complete indicator.
        Int32(4)
            Length of message contents in bytes, including self.

    """

    def __init__(self, server_response):
        Message.__init__(self, server_response)

class PasswordMessage(Message):
    """
    Definition from
    http://www.postgresql.org/docs/7.4/interactive/protocol-message-formats.html

    PasswordMessage (F)
        Byte1('p')
            Identifies the message as a password response.
        Int32
            Length of message contents in bytes, including self.
        String
            The password (encrypted, if requested).
    """

    def __init__(self, auth_type, password, username, salt=None):
        Message.__init__(self)
        self.auth_type = auth_type
        self.password = password
        self.username = username
        self.salt = salt
        self.prepareForServer()

    def prepareForServer(self):
        """Create a password message.

        Message From: Frontend
        Requires:
            auth_type:
                3 - Clear Text
                4 - Crypt
                5 - MD5
            password:  string
            salt:  string
        Returns:
            Byte1('p'):  Password Response
            Int32:  Length of message contents in bytes, including self.
            String:  The password(encrypted, if requested).
        """
        auth_type = self.auth_type
        username = self.username
        password = self.password
        salt = self.salt
        assert auth_type in (3, 4, 5), 'not a valid auth_type'
        if auth_type == 3:
            password = password + chr(0)    
            msglen = Int32(4 + len(password))
            message = 'p' + msglen + password
        elif auth_type == 5:
            assert username is not None, 'username should not be None.'
	    if oldmd5:
		passdigest = md5.new(password + username).hexdigest()
		digest = 'md5' + md5.new(passdigest + salt).hexdigest() + chr(0)
	    else:
		passdigest = hashlib.md5(password + username).hexdigest()
		digest = 'md5' + hashlib.md5(passdigest + salt).hexdigest() + chr(0)
            msglen = Int32(4 + len(digest))
            message = 'p' + msglen + digest
        else:
            raise NotImplementedError, 'Auth type not supported.'
        self.server_message = message
class CancelRequest(Message):
    """CancelRequest (F)

    Int32(16)

        Length of message contents in bytes, including self.
    Int32(80877102)

        The cancel request code. The value is chosen to contain 1234 in the most significant 16 bits, and 5678 in the least 16 significant bits. (To avoid confusion, this code must not be the same as any protocol version number.)
    Int32

        The process ID of the target backend.
    Int32

        The secret key for the target backend.
    """
    def __init__(self,pid,secret):
        Message.__init__(self)
        self.pid = pid
        self.secret = secret
        self.prepareForServer()
    def prepareForServer(self):
        msg = Int32(16) + Int32(80877102) + Int32(self.pid) + Int32(self.secret)
        self.server_message = msg
        
class Query(Message):
    """Simple Query
    Definition from
    http://www.postgresql.org/docs/7.4/interactive/protocol-message-formats.html

    Query (F)
        Byte1('Q')
            Identifies the message as a simple query.
        Int32
            Length of message contents in bytes, including self.
        String
            The query string itself.

    """

    def __init__(self, sqlquery):
        Message.__init__(self)
        self.sqlquery = sqlquery
        self.prepareForServer()

    def prepareForServer(self):
        sqlquery = self.sqlquery
        sqlquery = sqlquery + chr(0)
        length = len(sqlquery) + 4 # hm
        msg = 'Q' + Int32(length) + sqlquery
        self.server_message = msg

class ReadyForQuery(Message):
    """
    Definition from
    http://www.postgresql.org/docs/7.4/interactive/protocol-message-formats.html

    ReadyForQuery (B)
        Byte1('Z')
            Identifies the message type. ReadyForQuery is sent whenever the 
            backend is ready for a new query cycle.
        Int32(5)
            Length of message contents in bytes, including self.
        Byte1
            Current backend transaction status indicator. Possible values are 
            'I' if idle (not in a transaction block); 'T' if in a transaction 
            block; or 'E' if in a failed transaction block (queries will be 
            rejected until block is ended).

    """

    def __init__(self, server_response):
        Message.__init__(self, server_response)
        self.trasaction_status_indicator = None

    def prepareForClient(self):
        server_response = self.server_response
        self.transaction_status_indicator = server_response[5]

class RowDescription(Message):
    """
    Definition from
    http://www.postgresql.org/docs/7.4/interactive/protocol-message-formats.html

    RowDescription (B)
        Byte1('T')
            Identifies the message as a row description.
        Int32
            Length of message contents in bytes, including self.
        Int16
            Specifies the number of fields in a row (may be zero).
        Then, for each field, there is the following:
        String
            The field name.
        Int32
            If the field can be identified as a column of a specific table, 
            the object ID of the table; otherwise zero.
        Int16
            If the field can be identified as a column of a specific table, 
            the attribute number of the column; otherwise zero.
        Int32
            The object ID of the field's data type.
        Int16
            The data type size (see pg_type.typlen). Note that negative values 
            denote variable-width types.
        Int32
            The type modifier (see pg_attribute.atttypmod). The meaning of the 
            modifier is type-specific.
        Int16
            The format code being used for the field. Currently will be zero 
            (text) or one (binary). In a RowDescription returned from the 
            statement variant of Describe, the format code is not yet known 
            and will always be zero.

    """

    def __init__(self, server_response):
        Message.__init__(self, server_response)
        self.columns = None
        self.prepareForClient()

    def prepareForClient(self):
        server_response = self.server_response
        self.colnum = Int16U(server_response[5:7])
        columns = []
        self.columns = columns
        position = 7
        for colnum in range(self.colnum):
            column = {}
            portion = server_response[position:]
            endstring = position + portion.find(chr(0))
            column['name'] = server_response[position:endstring]
            pos1 = endstring + 1
            pos2 = pos1 + 4
            column['oid'] = Int32U(server_response[pos1:pos2])
            pos1 += 4
            pos2 = pos1 + 2
            column['attnum'] = Int16U(server_response[pos1:pos2])
            pos1 += 2
            pos2 = pos1 + 4
            column['type_oid'] = Int32U(server_response[pos1:pos2])
            pos1 += 4
            pos2 = pos1 + 2
            column['typelen'] = Int16U(server_response[pos1:pos2])
            pos1 += 2
            pos2 = pos1 + 4
            column['atttypmod'] = Int32U(server_response[pos1:pos2])
            pos1 += 4
            pos2 = pos1 + 2
            column['format_code'] = Int16U(server_response[pos1:pos2])
            position = endstring + 19
            columns.append(column)

class StartupMessage(Message):
    """
    Definition from
    http://www.postgresql.org/docs/7.4/interactive/protocol-message-formats.html

    StartupMessage (F)
        Int32
            Length of message contents in bytes, including self.
        Int32(196608)
            The protocol version number. The most significant 16 bits are the 
            major version number (3 for the protocol described here). The 
            least significant 16 bits are the minor version number (0 for the 
            protocol described here).
        The protocol version number is followed by one or more pairs of 
        parameter name and value strings. A zero byte is required as a 
        terminator after the last name/value pair. Parameters can appear in 
        any order. user is required, others are optional. Each parameter is 
        specified as:
        String
            The parameter name. Currently recognized names are:
            user
                The database user name to connect as. Required; there is no 
                default.
            database
                The database to connect to. Defaults to the user name.
            options
                Command-line arguments for the backend. (This is deprecated in 
                favor of setting individual run-time parameters.)
            In addition to the above, any run-time parameter that can be set 
            at backend start time may be listed. Such settings will be applied 
            during backend start (after parsing the command-line options if 
            any). The values will act as session defaults.
        String
            The parameter value.
    """

    def __init__(self, user, database=None):
        Message.__init__(self)
        self.user = user
        self.database = database
        self.prepareForServer()

    def prepareForServer(self):
        user = self.user
        database = self.database
        protocol_major = Int16(3)
        protocol_minor = Int16(0)
        cnstring = 'user' + chr(0) + user + chr(0)
        cnstring += 'database' + chr(0) + database + chr(0) + chr(0)
        # Message Length
        # 4 is for length of length number.
        strlen = 4 + len(cnstring) + len(protocol_major) + len(protocol_minor)
        strlen = Int32(strlen)
        self.server_message = strlen + protocol_major + protocol_minor
        self.server_message += cnstring

class Terminate(Message):
    """
    Definition from
    http://www.postgresql.org/docs/7.4/interactive/protocol-message-formats.html

    Terminate (F)
        Byte1('X')
            Identifies the message as a termination.
        Int32(4)
            Length of message contents in bytes, including self.

    """

    def __init__(self):
        Message.__init__(self)
        self.prepareForServer()

    def prepareForServer(self):
        self.server_message = 'X' + Int32(4)

def createServerMessage(server_response):
    """Determines message type and return message from server_response."""
    identity = server_response[0]
    message_class = None
    if identity == 'R':
        # Authentication Request
        length = Int32U(server_response[1:5])
        # Specifier determines type of authentication.
        specifier = Int32U(server_response[5:9])
        specifiers = {0 :AuthenticationOk,
                      1 :AuthenticationKerberosV4,
                      2 :AuthenticationKerberosV5,
                      3 :AuthenticationCleartextPassword,
                      4 :AuthenticationCryptPassword,
                      5 :AuthenticationMD5Password,
                      6 :AuthenticationSCMCredential        }
        message_class = specifiers[specifier]
    elif identity == 'Z':
        message_class = ReadyForQuery
    elif identity == 'S':
        message_class = ParameterStatus
    elif identity == 'K':
        message_class = BackendKeyData
    elif identity == 'C':
        message_class = CommandComplete
    elif identity == 'G':
        message_class = CopyInResponse
    elif identity == 'H':
        message_class = CopyOutResponse
    elif identity == 'T':
        message_class = RowDescription
    elif identity == 'D':
        message_class = DataRow
    elif identity == 'I':
        message_class = EmptyQueryResponse
    elif identity == 'E':
        message_class = ErrorResponse
    elif identity == 'N':
        message_class = NoticeResponse
    elif identity == '1':
        message_class = ParseComplete
    if message_class is None:
        raise Exception, "Can't determine message type."
    else:
        return message_class(server_response)

