""" module to replace PyGreSQL _pg.so

    LICENSE
    =======

    Copyright (c) 2004, Randall Smith
    All rights reserved.

    Redistribution and use in source and binary forms, with or without 
    modification, are permitted provided that the following conditions are met:

        * Redistributions of source code must retain the above copyright 
          notice, this list of conditions and the following disclaimer.
        * Redistributions in binary form must reproduce the above copyright 
          notice, this list of conditions and the following disclaimer in the 
          documentation and/or other materials provided with the distribution.
        * Neither the name of the <ORGANIZATION> nor the names of its 
          contributors may be used to endorse or promote products derived from 
          this software without specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
    AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
    IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
    ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
    LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
    CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
    SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
    CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
    ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
    POSSIBILITY OF SUCH DAMAGE.

    NOTES
    =======
    This is alpha software.  Don't use it on valuable data.

    It replaces the _pg.so module included with PyGreSQL to make this a 100%
    pure Python database module.  I created as an experiment to determine the
    feasability of creating pure Python drivers for databases.

"""

import os
import socket
import pgmessages
from PGSocket import PGSocket

RESULT_DQL = 4

class Connection(object):

    def __init__(self, host, port, dbname, user, passwd):
        # If we are missing a username, try to get it.
        if user is None:
            try:
                user = os.getlogin()
            except:
                raise Exception, 'user is required.'
        if dbname is None:
            dbname = user
        self.host = host
        self.port = int(port)
        self.dbname = dbname
        self.user = user
        self.passwd = passwd
        self.params = {}
        self.dbdata={}
        self.dbdata["host"]=self.host
        self.dbdata["port"]=self.port
        self.dbdata["dbname"]=self.dbname
        self.dbdata["user"]=self.user
        self.dbdata["passwd"]=self.passwd
        self.error = False
        
        self.connect()
    def reconnect(self):
        # attempt to restore the connection, only for tcp
        # return a new PGSocket
        print "trying reconnect"
        self._sock.close()
        if self.host is not None:
            try:
                sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
            except socket.error, msg:
                self.error = "%s \n%s"%("Could not create socket",msg)
                return False
            try:
                # Set timeout so we don't hang forever if something goes wrong.
                sock.settimeout(self.TIMEOUT)
                sock.connect((self.host, self.port))
            except socket.error, msg:
                self.error = "%s %s:%s \n%s"%("Could not connect to",self.host,self.port,msg)
                return False
        self._sock = sock
        startup_message = pgmessages.StartupMessage(self.user, self.dbname)
        sock.send(startup_message.server_message)
        # Now that the startup message is sent, use our custom socket.
        pgsock = PGSocket(sock)
        self.pgsock = pgsock
        message = pgsock.getMessage()
        if message.format_name == 'ErrorResponse':
            self.error = 'got an error.  Probably IDENT failure'
            return False
        if isinstance(message, pgmessages.AuthRequestMessage):
            password_type = message.specifier
            salt = message.salt
            msg = pgmessages.PasswordMessage(password_type, self.passwd,
                                             self.user, salt)
            pgsock.send(msg.server_message)
        nextmsg = pgsock.getMessage()
        if nextmsg.format_name == 'ErrorResponse':
            self.error = 'Authentication Failed.'
            return False
        elif nextmsg.format_name == 'AuthenticationOk':
            pass
        elif nextmsg.format_name == 'ParameterStatus':
            # Must have used ident.
            pass
        else:
            self.error = 'Did not handle response, %s' % nextmsg.format_name
            return False
        while nextmsg.format_name != 'ReadyForQuery':
            if nextmsg.format_name == 'ErrorResponse':
                # something went wrong, db doesn't exists on server ?
                self.error = nextmsg.param_value
                return False
            if nextmsg.format_name == 'ParameterStatus':
                self.params[nextmsg.param_name] = nextmsg.param_value
            if nextmsg.format_name == 'BackendKeyData':
                self.pid = nextmsg.process_id
                self.secret = nextmsg.secret_key        
                self.dbdata["pid"]=nextmsg.process_id
                self.dbdata["secret"] = nextmsg.secret_key  
            nextmsg = pgsock.getMessage()

        return pgsock
            
    def connect(self):
        # If host is provided and Unix sockets available, do so.
        TIMEOUT=120
        self.TIMEOUT = TIMEOUT
        if self.host is not None:
            try:
                sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
            except socket.error, msg:
                self.error = "%s \n%s"%("Could not create socket",msg)
                return
            try:
                # Set timeout so we don't hang forever if something goes wrong.
                sock.settimeout(TIMEOUT)
                sock.connect((self.host, self.port))
            except socket.error, msg:
                self.error = "%s %s:%s \n%s"%("Could not connect to",self.host,self.port,msg)
                return
        elif self.host is None and socket.AF_UNIX:
            try:
                try:
                    sock = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
                except socket.error, msg:
                    self.error = "%s \n%s"%("Could not create socket",msg)
                    return
                try:
                    # Set timeout so we don't hang forever if something goes wrong.
                    sock.settimeout(TIMEOUT)
                    sock.connect('/var/run/postgresql/.s.PGSQL.5432')
                except socket.error, msg:
                    self.error = "%s \n%s"%("Could not connect to /var/run/postgresql/.s.PGSQL.5432",msg)
                    return
            except:
                self.host = 'localhost'
                try:
                    sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
                except socket.error, msg:
                    self.error = "%s \n%s"%("Could not create socket",msg)
                    return
        else: 
            self.host = 'localhost'
            try:
                sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
            except socket.error, msg:
                self.error = "%s \n%s"%("Could not create socket",msg)
                return
            try:
                # Set timeout so we don't hang forever if something goes wrong.
                sock.settimeout(TIMEOUT)
                sock.connect((self.host, self.port))
            except socket.error, msg:
                self.error = "%s %s:%s \n%s"%("Could not connect to",self.host,self.port,msg)
                return
        
        self._sock = sock
        startup_message = pgmessages.StartupMessage(self.user, self.dbname)
        sock.send(startup_message.server_message)
        # Now that the startup message is sent, use our custom socket.
        pgsock = PGSocket(sock)
        self.pgsock = pgsock
        message = pgsock.getMessage()
        if message.format_name == 'ErrorResponse':
            self.error = 'got an error.  Probably IDENT failure'
            return
        if isinstance(message, pgmessages.AuthRequestMessage):
            password_type = message.specifier
            salt = message.salt
            msg = pgmessages.PasswordMessage(password_type, self.passwd,
                                             self.user, salt)
            pgsock.send(msg.server_message)
        nextmsg = pgsock.getMessage()
        if nextmsg.format_name == 'ErrorResponse':
            self.error = 'Authentication Failed.'
            return
        elif nextmsg.format_name == 'AuthenticationOk':
            pass
        elif nextmsg.format_name == 'ParameterStatus':
            # Must have used ident.
            pass
        else:
            self.error = 'Did not handle response, %s' % nextmsg.format_name
            return
        while nextmsg.format_name != 'ReadyForQuery':
            if nextmsg.format_name == 'ErrorResponse':
                # something went wrong, db doesn't exists on server ?
                self.error = nextmsg.param_value
                return
            if nextmsg.format_name == 'ParameterStatus':
                self.params[nextmsg.param_name] = nextmsg.param_value
            if nextmsg.format_name == 'BackendKeyData':
                self.pid = nextmsg.process_id
                self.secret = nextmsg.secret_key        
                self.dbdata["pid"]=nextmsg.process_id
                self.dbdata["secret"] = nextmsg.secret_key  
            nextmsg = pgsock.getMessage()

    def source(self):
        return Source(self)

    def close(self):
        self.pgsock.send(pgmessages.Terminate().server_message)
        self._sock.close()

class Source(object):
    def __init__(self, parent):
        self.parent = parent
        self.resulttype = 0
        self.ntuples = 0
        self.column_info = None
        self.__resultpos = 0
        

    def execute(self, query):
        """Execute a simple query.
        
        Return the number of rows affected or None for a query that actually
        returns rows.        
        """
        reconnected = False
        self.error = False
        pgsock = self.parent.pgsock
        querymsg = pgmessages.Query(query).server_message
        # This could be a problem at any time !! VVVVVVVVV
        try:
            pgsock.send(querymsg)
        except:
            print "FAILED TO SEND QUERY"
            # Attempt to restore connection, this is a most likely place after some "idle" time
            self.pgsock = self.parent.reconnect()
            pgsock = self.pgsock
            if pgsock == False:return False
            pgsock.send(querymsg)
            print "Resending:%s"%querymsg
            reconnected = True
        results = []
        try:
            nextmsg = pgsock.getMessage()
        except:
            print "FAILED TO RECEIVE QUERY"
        if nextmsg==False:
            self.pgsock = self.parent.reconnect()
            pgsock = self.pgsock
            if pgsock == False:return False
            pgsock.send(querymsg)
            print "Resending:%s"%querymsg
            reconnected = True
        if nextmsg.format_name == 'RowDescription':
            self.column_info = {}
            self.column_info['count'] = nextmsg.colnum
            self.column_info['attributes'] = nextmsg.columns
        if nextmsg.format_name == 'ErrorResponse':
            # We have encountered an error
            self.error = True
            self.errormessage = nextmsg.server_response[5:]
        while nextmsg.format_name <> 'ReadyForQuery':
            #print "Am I looping/stuk ?"
            try:
                nextmsg = pgsock.getMessage()
            except:
                print "FAILED TO RECEIVE QUERY IN LOOP"
            if nextmsg==False:
                print "FAILD TO RECEIVE QUERY IN LOOP (False)"
                self.pgsock = self.parent.reconnect()
                pgsock = self.pgsock
                if pgsock == False:return False
                pgsock.send(querymsg)
                print "Resending:%s"%querymsg
                reconnected = True
                nextmsg = pgsock.getMessage()
                if nextmsg.format_name == 'RowDescription':
                    self.column_info = {}
                    self.column_info['count'] = nextmsg.colnum
                    self.column_info['attributes'] = nextmsg.columns
                    nextmsg = pgsock.getMessage()
            elif nextmsg.format_name == 'DataRow':
                results.append(nextmsg.rowdata)

        # Fix this later.
        self._rset = results
        self.ntuples = len(results)
        self.resulttype = 4
        self.__resultpos = 0
        return None

    def listinfo(self):
        """
		listinfo is a sequence of
		(index, column_name, type_oid)
		getdescr returns all items needed for a
		description tuple except the column_name.

        Appears to be a tuple of tuples.  The index is just an incrementing
        number starting with zero.
        """
        if self.column_info is not None:
            info = []
            columns = self.column_info['attributes']
            column_count = len(columns)
            for i in range(column_count):
                column = columns[i]
                column_info = (i, column['name'], column['type_oid'])
                info.append(column_info)
            return tuple(info)
        else:
            return ()

    def fetch(self, numrows):
        """Return results.
       
        -1 All Rows
        """
        if numrows == -1:
            subset = self._rset[self.__resultpos:]
            self.__resultpos = self.ntuples
        elif numrows > self.ntuples - self.__resultpos:
            subset = self._rset[self.__resultpos:]
            self.__resultpos = self.ntuples
        else:
            newpos = self.__resultpos + numrows
            subset = self._rset[self.__resultpos:newpos]
            self.__result_pos = newpos
        return subset
		
def connect(dbbase, dbhost, dbport, dbpot, dbtty, dbuser, dbpasswd):
    if dbport == -1:
        dbport = 5432
    return Connection(dbhost, dbport, dbbase, dbuser, dbpasswd)
   

